/*
 * Generic alert subsystem
 */
#include <time.h>

#include "libfma.h"
#include "lf_alert.h"

#include "fms.h"
#include "fms_alert.h"
#include "fms_error.h"
#include "fms_notify.h"

/*
 * Local routines
 */
static void fms_register_alert_event(struct fms_alert *ap);
static void fms_relic_alert_event(struct fms_alert *ap);
static void fms_ack_alert_event(struct fms_alert *ap);

/*
 * initialize alert variables
 */
int
init_alert_vars()
{
  struct fms_alert_vars *av;

  LF_CALLOC(av, struct fms_alert_vars, 1);
  LF_CALLOC(av->alerts_anchor, struct fms_alert, 1);

  /* set alerts list to be empty */
  fms_init_alert_anchor(av->alerts_anchor);

  F.alert_vars = av;

  av->next_alert_id = 1;	/* start at 1 because we can */

  return 0;

 except:
  return -1;
}


/*
 * Initialize alert subsystem
 */
int
init_alerts()
{
  struct fms_alert_vars *av;

  av = F.alert_vars;

  /*
   * If specified, set up for alerts to go to a user program
   */
  {
    char *prog;
    int id;

    prog = getenv("FMS_ALERT_EXEC");
    if (prog != NULL && *prog != '\0') {
      id = register_notify_exec(FMS_EVENT_ALERT, prog);
      if (id == -1) {
	fprintf(stderr, "Cannot set up alert exec client\n");
	exit(1);
      }
    }
  }

  return 0;
}

/*
 * Allocate an alert - this returns an alert struct the
 * unique ID filled in.
 */
struct fms_alert *
fms_allocate_alert(
  enum lf_alert_type type)
{
  struct fms_alert *ap;
  struct fms_alert_vars *av;

  av = F.alert_vars;

  /* allocate the alert and  fill in unique ID */
  LF_CALLOC(ap, struct fms_alert, 1);
  ap->alert.alert_id = av->next_alert_id;
  ap->alert.alert_type = type;

  ++av->next_alert_id;

  /* initialize pointers to "no list membership" */
  fms_init_alert_anchor(ap);

  return ap;

 except:
  return NULL;
}

/*
 * Register an alert whose data fields have been filled in by
 * the registrar.  The timestamp is filled in now.
 */
void
fms_register_alert(
  struct fms_alert *ap)
{
  struct fms_alert_vars *av;
  time_t t;

  av = F.alert_vars;

  t = time(0);
  ap->alert.alert_time = t;

  /* link this alert into the chain */
  fms_main_link_alert(av->alerts_anchor, ap);

  /* log an event associated with this alert */
  fms_register_alert_event(ap);
}

/*
 * Remove this alert from all lists and free it.
 */
void
fms_cancel_alert(
  struct fms_alert *ap)
{
  fms_subj_unlink_alert(ap);
  fms_main_unlink_alert(ap);
  LF_FREE(ap);
}

/*
 * Alert has been acknowledged by an operator.  This takes an alert ID
 * structure since that is all an outside party will have.
 * Record this and cancel the alert if it is a relic or only needs an ACK.
 */
int
fms_ack_alert(
  uint32_t id)
{
  struct fms_alert *ap;

  /* any such alert? */
  ap = fms_find_alert(id);
  if (ap == NULL) return -1;

  ap->alert.acked = TRUE;

  /* log an event associated with this alert */
  fms_ack_alert_event(ap);

  /* If is a relic or needs ACK only, cancel it */
  if (ap->alert.relic || (lf_alert_flags(&(ap->alert)) & LF_AF_ACK_ONLY) != 0) {
    fms_cancel_alert(ap);
  }
  return 0;
}

/*
 * The condition which caused an alert is gone.  Make the alert a relic,
 * possibly cancelling it altogether.
 */
void
fms_relic_alert(
  struct fms_alert *ap)
{
  ap->alert.relic = TRUE;

  /* log an event that this alert is being reliced */
  fms_relic_alert_event(ap);

  /* If alert does not need an ack or it has gotten one, remove it */
  if (ap->alert.acked || (lf_alert_flags(&(ap->alert)) & LF_AF_NEED_ACK) == 0) {
    fms_cancel_alert(ap);
  }
}

/*
 * Find an alert in our list given an ID struct
 */
struct fms_alert *
fms_find_alert(
  uint32_t id)
{
  struct fms_alert *ap;
  struct fms_alert *anchor;

  /* start out from "next" of anchor */
  anchor = F.alert_vars->alerts_anchor;
  ap = anchor->main_next;

  /* scan looking for matching ID */
  while (ap != anchor) {
    if (ap->alert.alert_id == id) {
      return ap;
    }
    ap = ap->main_next;
  }
  return NULL;
}

/*
 * Given a list of alerts and an array of types, relic all alerts in the
 * list that match any type in the array
 */
void
fms_alert_relic_matches_in_subj_list(
  struct fms_alert *anchor,
  lf_alert_type_t *types,
  int nt)
{
  struct fms_alert *ap;
  struct fms_alert *cap;
  int t;

  /* scan through entire list */
  ap = anchor->subj_next;
  while (ap != anchor) {

    /* save current and get next now since list may change */
    cap = ap;
    ap = ap->subj_next;

    if (cap->alert.relic) continue;	/* already relic, continue */

    /* compare to each type and relic if match */
    for (t=0; t<nt; ++t) {
      if (cap->alert.alert_type == types[t]) {
	fms_relic_alert(cap);	/* this may destroy cap */
	break;
      }
    }
  }
}

/*
 * Cancel all the alerts in a subject list
 */
void
fms_cancel_subject_alerts(
  struct fms_alert *anchor)
{
  struct fms_alert *ap;
  struct fms_alert *nap;

  /* scan through entire subject list, cancelling each one */
  ap = anchor->subj_next;
  while (ap != anchor) {
    nap = ap->subj_next;	/* save next since cancel free()s ap */
    fms_cancel_alert(ap);

    ap = nap;
  }
}

/*
 * Perform a notify that an alert has been registered
 */
static void
fms_register_alert_event(
  struct fms_alert *ap)
{
  lf_string_t buf;
  struct lf_alert *lap;
  char *bp;

  lap = &(ap->alert);

  bp = buf;
  bp += sprintf(bp, "%s ", lf_alert_name(lap));
  bp += lf_alert_string(bp, lap);
  fms_notify(FMS_EVENT_ALERT, buf);
}

/*
 * Perform a notify that an alert has been reliced
 */
static void
fms_relic_alert_event(
  struct fms_alert *ap)
{
  lf_string_t buf;

  sprintf(buf, "relic event type %s", lf_alert_name(&(ap->alert)));
  fms_notify(FMS_EVENT_INFO, buf);
}

/*
 * Perform a notify that an alert has been ACKed
 */
static void
fms_ack_alert_event(
  struct fms_alert *ap)
{
  lf_string_t buf;

  sprintf(buf, "ACK event type %s", lf_alert_name(&(ap->alert)));
  fms_notify(FMS_EVENT_INFO, buf);
}
